<?php

if (!defined('ABSPATH')) {
    exit; // Seguridad para evitar accesos directos
}

// Incluir archivos necesarios
require_once plugin_dir_path(__FILE__) . 'StarterAI_Config.php';
require_once plugin_dir_path(__FILE__) . 'StarterAI_Logger.php';

class StarterAI
{

    // Inicializar la clase y registrar eventos
    public static function init()
    {

        //Acciones para obtener datos del carrito
        self::front_display_add_cart();
        self::front_display_update_cart();

        //Acciones para obtener datos del checkout en tiempo real
        self::ensure_cart_token();
        self::track_checkout_fields();
        self::handle_checkout_tracking();

        //Acciones para obtener cuando se cree la orden
        self::track_order_updates();

        //Acciones para actualizar url con carrito abandonado
        self::save_cart_data();  // Guardar carrito cuando se actualiza
        self::recover_cart();    // Recuperar carrito cuando se accede a la URL

        self::restore_cart_token(); // Restaurar el carrito si se accede desde un link

    }

    // 1️⃣ Registrar cuando se agrega un producto al carrito
    public static function front_display_add_cart()
    {
        add_action('woocommerce_add_to_cart', function ($cart_item_key, $product_id, $quantity, $variation_id, $cart) {

            // 🔹 Evitar ejecución en la URL de carrito abandonado
            if (isset($_GET['recover_cart']) && !empty($_GET['recover_cart'])) {
                error_log("🚫 Se evitó la ejecución de add_cart en recover_cart.");
                return;
            }

            // Obtener el cart_token de la sesión
            $cart_token = StarterAI::get_cart_token();

            // Forzar el recalculo de los totales
            WC()->cart->calculate_totals(); // Recalcular totales del carrito

            // Obtener los productos en el carrito
            $cart_contents = WC()->cart->get_cart();

            $data = [];
            foreach ($cart_contents as $cart_item) {
                $product_id   = $cart_item['product_id'];
                $product      = wc_get_product($product_id);
                $unit_price   = $product ? floatval($product->get_price()) : 0; // Obtener precio unitario
                $subtotal     = $unit_price * $cart_item['quantity']; // Calcular subtotal

                $data[] = [
                    'product_id'   => $product_id,
                    'product_name' => get_the_title($product_id),
                    'quantity'     => $cart_item['quantity'],
                    'subtotal'     => $subtotal,
                    'unit_price'   => $unit_price
                ];
            }

            // Guardar el carrito en la base de datos
            self::store_cart(); // 🔥 AQUÍ SE GUARDA EL CARRITO

            // Verificar si StarterAI está cargado antes de llamar a la función
            if (class_exists('StarterAI') && method_exists('StarterAI', 'get_abandoned_cart_url')) {
                $recovery_url = StarterAI::get_abandoned_cart_url($cart_token);
                error_log("🔗 URL generada correctamente: " . $recovery_url);
            } else {
                error_log("❌ ERROR: La función get_abandoned_cart_url() no está disponible.");
                $recovery_url = '';
            }

            // 🔹 Obtener la URL de la API con el token
            $api_url = StarterAI_Config::get_api_endpoint('cart_update');

            if ($api_url) {
                // 🔥 Enviar datos a la API
                StarterAI_Logger::send_data_to_endpoint('cart_update', [
                    'event'      => 'add_to_cart',
                    'cart_token' => $cart_token,
                    'cart_items' => $data,
                    'cart_url' => $recovery_url
                ]);
            } else {
                error_log("❌ No se encontró una URL válida para cart_update.");
            }

            // Registrar logs si está habilitado
            if (get_option('avw_habilitar_logs', 'no') === 'yes') {
                wc_get_logger()->info(
                    'PRODUCTO AGREGADO AL CARRO :: /CartRegistro :: ' . json_encode([
                        'event'      => 'add_to_cart',
                        'cart_token' => $cart_token,
                        'cart_items' => $data,
                        'recovery_url' => $recovery_url
                    ]),
                    ['source' => 'asistentevirtualwoocommerce']
                );
            }
        }, 10, 5);
    }

    // 2️⃣ Registrar cuando se actualiza el carrito
    public static function front_display_update_cart()
    {
        add_action('woocommerce_cart_updated', function () {

            // 🔹 Evitar ejecución en la URL de carrito abandonado
            if (isset($_GET['recover_cart']) && !empty($_GET['recover_cart'])) {
                error_log("🚫 Se evitó la ejecución de update_cart en recover_cart.");
                return;
            }

            $cart_token = StarterAI::get_cart_token();

            // Obtener los productos en el carrito
            $cart_contents = WC()->cart->get_cart();

            $data = [];
            foreach ($cart_contents as $cart_item) {
                $product_id   = $cart_item['product_id'];
                $product      = wc_get_product($product_id);
                $unit_price   = $product ? floatval($product->get_price()) : 0; // Obtener precio unitario
                $subtotal     = $unit_price * $cart_item['quantity']; // Calcular subtotal

                $data[] = [
                    'product_id'   => $product_id,
                    'product_name' => get_the_title($product_id),
                    'quantity'     => $cart_item['quantity'],
                    'subtotal'     => $subtotal,
                    'unit_price'   => $unit_price
                ];
            }

            // Verificar si StarterAI está cargado antes de llamar a la función
            if (class_exists('StarterAI') && method_exists('StarterAI', 'get_abandoned_cart_url')) {
                $recovery_url = StarterAI::get_abandoned_cart_url($cart_token);
                error_log("🔗 URL generada correctamente: " . $recovery_url);
            } else {
                error_log("❌ ERROR: La función get_abandoned_cart_url() no está disponible.");
                $recovery_url = '';
            }

            // 🔹 Obtener la URL de la API con el token
            $api_url = StarterAI_Config::get_api_endpoint('cart_update');

            if ($api_url) {
                // 🔥 Enviar datos a la API
                StarterAI_Logger::send_data_to_endpoint('cart_update', [
                    'event'      => 'add_to_cart',
                    'cart_token' => $cart_token,
                    'cart_items' => $data,
                    'cart_url' => $recovery_url
                ]);
            } else {
                // Registrar logs si está habilitado
                if (get_option('avw_habilitar_logs', 'no') === 'yes') {
                    wc_get_logger()->info(
                        'Error al registrar datos del carrito update cart'
                    );
                }
                //error_log("❌ No se encontró una URL válida para cart_update.");
            }

            // Registrar logs si está habilitado
            if (get_option('avw_habilitar_logs', 'no') === 'yes') {
                wc_get_logger()->info(
                    'PRODUCTO ACTUALIZADO EN EL CARRO PERSISTENTE :: /CartRegistro :: ' . json_encode([
                        'event'      => 'cart_updated',
                        'cart_token' => $cart_token,
                        'recovery_url' => $recovery_url,
                        'cart_items' => $data
                    ]),
                    ['source' => 'asistentevirtualwoocommerce']
                );
            }
        }, 10);
    }

    // 1️⃣ Crear `cart_token` en sesión y cookies
    public static function ensure_cart_token()
    {
        add_action('wp', function () {
            if (!WC()->session) {
                return;
            }

            $cart_token = WC()->session->get('cart_token');

            // 🔹 Si ya existe un `cart_token`, aseguramos que esté en la cookie y terminamos
            if (!empty($cart_token)) {
                if (!isset($_COOKIE['cart_token']) || $_COOKIE['cart_token'] !== $cart_token) {
                    setcookie('cart_token', $cart_token, time() + (86400 * 30), "/");
                    error_log("✅ Cart Token sincronizado con la cookie: " . $cart_token);
                }
                return; // 🔥 No seguir ejecutando el código innecesariamente
            }

            // 🔹 Restaurar `cart_token` desde la cookie si existe
            if (isset($_COOKIE['cart_token']) && !empty($_COOKIE['cart_token'])) {
                $cart_token = $_COOKIE['cart_token'];
                WC()->session->set('cart_token', $cart_token);
                error_log("♻️ Cart Token restaurado desde la cookie: " . $cart_token);
                return;
            }

            // 🔹 SOLO si no hay `cart_token` en sesión ni en cookies, generamos uno nuevo
            $cart_token = uniqid('cart_', true);
            WC()->session->set('cart_token', $cart_token);
            setcookie('cart_token', $cart_token, time() + (86400 * 30), "/");
            error_log("🆕 Nuevo Cart Token generado: " . $cart_token);
        });
    }

    public static function track_checkout_fields()
    {
        add_action('wp_footer', function () {
            if (!is_checkout()) return;
?>
            <script type="text/javascript">
                jQuery(document).ready(function($) {
                    console.log("✅ Script de seguimiento en checkout cargado correctamente.");

                    function getCheckoutFieldsData() {
                        const data = {
                            action: "track_checkout_data",
                            cart_token: "<?php echo esc_js(WC()->session->get('cart_token')); ?>"
                        };

                        $("input, select, textarea").each(function() {
                            let name = $(this).attr("name") || $(this).attr("id");
                            let value = $(this).val();

                            if (!name || !value) return;

                            // Normalizar nombre
                            name = name.replace(/-/g, "_");

                            // Solo si es billing o shipping
                            if (
                                name.startsWith("billing_") || name.startsWith("shipping_") ||
                                name === "email"
                            ) {
                                data[name] = value;
                            }
                            // if (name.startsWith("billing_") || name.startsWith("shipping_")) {
                            //     data[name] = value;
                            // }
                        });

                        console.log("📦 Datos capturados del checkout:", data);
                        return data;
                    }

                    function enviarDatosCheckout() {
                        const checkoutData = getCheckoutFieldsData();

                        const tieneDatos = Object.keys(checkoutData).some(key =>
                            key !== "action" && key !== "cart_token" && checkoutData[key] !== ""
                        );

                        if (tieneDatos) {
                            $.post('<?php echo admin_url('admin-ajax.php'); ?>', checkoutData, function(response) {
                                console.log("📬 Respuesta del servidor:", response);
                            }).fail(function(xhr, status, error) {
                                console.error("❌ Error al enviar los datos:", error);
                            });
                        } else {
                            console.warn("⚠️ Datos vacíos, no se envía al servidor.");
                        }
                    }

                    // Captura inicial
                    enviarDatosCheckout();

                    // Cuando se actualiza dinámicamente el checkout
                    $(document.body).on('updated_checkout', function() {
                        console.log("🔄 Checkout actualizado dinámicamente");
                        setTimeout(enviarDatosCheckout, 500);
                    });

                    // Cuando el usuario cambia cualquier campo
                    $('body').on('change', 'input, select, textarea', function() {
                        enviarDatosCheckout();
                    });
                });
            </script>
<?php
        });
    }

    // public static function track_checkout_fields()
    // {
    //     add_action('wp_footer', function () {
    //         echo '<script type="text/javascript">
    //             jQuery(document).ready(function($) {
    //                 console.log("🚀 Script de seguimiento en checkout cargado.");

    //                 function capturarCamposCheckout() {
    //                     var checkoutData = {
    //                         action: "track_checkout_data",
    //                         cart_token: "' . esc_js(WC()->session->get("cart_token")) . '",
    //                         first_name: $("#billing_first_name").val() || "",
    //                         last_name: $("#billing_last_name").val() || "",
    //                         email: $("#billing_email").val() || "",
    //                         phone: $("#billing_phone").val() || "",
    //                         address: $("#billing_address_1").val() || "",
    //                         city: $("#billing_city").val() || "",
    //                         state: $("#billing_state").val() || "",
    //                         postcode: $("#billing_postcode").val() || "",
    //                         country: $("#billing_country").val() || ""
    //                     };

    //                     console.log("📌 Datos capturados del checkout:", checkoutData);

    //                     // Enviar datos por AJAX solo si el checkout ha sido actualizado completamente
    //                     if (checkoutData.first_name || checkoutData.email) {
    //                         $.post("' . admin_url('admin-ajax.php') . '", checkoutData, function(response) {
    //                             console.log("✅ Datos enviados al servidor:", response);
    //                         }).fail(function(xhr, status, error) {
    //                             console.error("❌ Error en la petición AJAX:", error);
    //                         });
    //                     }
    //                 }

    //                 // Capturar eventos de cambio en los campos
    //                 $("input, select, textarea").on("change", function() {
    //                     console.log("✏️ Cambio detectado:", $(this).attr("name"), "Nuevo valor:", $(this).val());
    //                     capturarCamposCheckout();
    //                 });

    //                 // Ejecutar cuando WooCommerce actualiza dinámicamente el checkout
    //                 $(document.body).on("updated_checkout", function() {
    //                     console.log("🔄 Checkout actualizado dinámicamente...");
    //                     setTimeout(capturarCamposCheckout, 500); // Retraso para asegurar que WooCommerce termine de actualizar
    //                 });

    //                 // Captura inicial al cargar la página
    //                 setTimeout(capturarCamposCheckout, 1000);
    //             });
    //         </script>';
    //     });
    // }

    // 3️⃣ Procesar los datos enviados por AJAX
    public static function handle_checkout_tracking()
    {
        add_action('wp_ajax_track_checkout_data', [__CLASS__, 'process_checkout_tracking']);
        add_action('wp_ajax_nopriv_track_checkout_data', [__CLASS__, 'process_checkout_tracking']);
    }

    // public static function process_checkout_tracking()
    // {

    //     $cart_token = sanitize_text_field($_POST['cart_token']);
    //     $checkout_data = [
    //         'first_name'  => sanitize_text_field($_POST['first_name']),
    //         'last_name'   => sanitize_text_field($_POST['last_name']),
    //         'email'       => sanitize_email($_POST['email']),
    //         'phone'       => sanitize_text_field($_POST['phone']),
    //         'address'     => sanitize_text_field($_POST['address']),
    //         'city'        => sanitize_text_field($_POST['city']),
    //         'state'       => sanitize_text_field($_POST['state']),
    //         'postcode'    => sanitize_text_field($_POST['postcode']),
    //         'country'     => sanitize_text_field($_POST['country']),
    //     ];

    //     // Registrar logs si está habilitado
    //     if (get_option('avw_habilitar_logs', 'no') === 'yes') {
    //         wc_get_logger()->info(
    //             'datos antes de enviar :: /Checkout :: ' . json_encode($checkout_data),
    //             ['source' => 'asistentevirtualwoocommerce']
    //         );
    //     }

    //     // Guardar los datos en la base de datos usando update_option
    //     update_option('avw_checkout_' . $cart_token, json_encode($checkout_data));

    //     // 🔹 Obtener la URL de la API con el token
    //     $api_url = StarterAI_Config::get_api_endpoint('checkout');

    //     // Registrar logs si está habilitado
    //     if (get_option('avw_habilitar_logs', 'no') === 'yes') {
    //         wc_get_logger()->info(
    //             'api url :: /Checkout :: ' . $api_url,
    //             ['source' => 'asistentevirtualwoocommerce']
    //         );
    //     }

    //     if ($api_url) {
    //         // 🔥 Enviar datos a la API
    //         StarterAI_Logger::send_data_to_endpoint('checkout', [
    //             'cart_token' => $cart_token,
    //             'contact_information' => $checkout_data
    //         ]);
    //     } else {
    //         // Registrar logs si está habilitado
    //         if (get_option('avw_habilitar_logs', 'no') === 'yes') {
    //             wc_get_logger()->info(
    //                 'Error al registrar datos del checkout'
    //             );
    //         }
    //         //error_log("❌ No se encontró una URL válida para cart_update.");
    //     }

    //     // Registrar logs si está habilitado
    //     if (get_option('avw_habilitar_logs', 'no') === 'yes') {
    //         wc_get_logger()->info(
    //             'Checkout actualizado :: /Checkout :: ' . json_encode([
    //                 'event'      => 'update_checkout',
    //                 'cart_token' => $cart_token,
    //                 'checkout' => $checkout_data
    //             ]),
    //             ['source' => 'asistentevirtualwoocommerce']
    //         );
    //     }

    //     wp_send_json_success();
    // }

    public static function process_checkout_tracking()
    {
        $cart_token = sanitize_text_field($_POST['cart_token'] ?? '');

        if (empty($cart_token)) {
            error_log("❌ No se recibió el cart_token en la petición AJAX.");
            wp_send_json_error(['message' => 'Falta el token del carrito']);
            return;
        }

        $checkout_data = [];
        foreach ($_POST as $key => $value) {
            // Solo capturamos campos que empiecen con billing_ o shipping_
            if (strpos($key, 'billing_') === 0 || strpos($key, 'shipping_') === 0) {
                $checkout_data[$key] = sanitize_text_field($value);
            }
        }

        if (empty($checkout_data)) {
            error_log("⚠️ No se capturaron datos válidos del checkout.");
            wp_send_json_error(['message' => 'Sin datos']);
            return;
        }

        // Guardar en la base de datos
        update_option('avw_checkout_' . $cart_token, json_encode($checkout_data));

        // Logs
        if (get_option('avw_habilitar_logs', 'no') === 'yes') {
            wc_get_logger()->info(
                '✅ Datos de checkout registrados :: ' . json_encode([
                    'cart_token'     => $cart_token,
                    'checkout_data'  => $checkout_data
                ]),
                ['source' => 'asistentevirtualwoocommerce']
            );
        }

        // Enviar a la API si está configurada
        $api_url = StarterAI_Config::get_api_endpoint('checkout');
        if (!empty($api_url)) {
            StarterAI_Logger::send_data_to_endpoint('checkout', [
                'cart_token'           => $cart_token,
                'contact_information'  => $checkout_data
            ]);
        }

        wp_send_json_success(['message' => 'Checkout procesado']);
    }


    public static function track_order_updates()
    {
        add_action(
            'woocommerce_order_status_changed',
            function ($order_id, $from_status, $to_status, $order) {
                // Si el pedido fue cancelado, no registramos el log
                if ($to_status == 'cancelled') {
                    return;
                }

                // 🔥 Obtener el cart_token desde los metadatos del pedido
                $cart_token = get_post_meta($order_id, '_cart_token', true);

                // 🔥 Si el `cart_token` está vacío, intentar obtenerlo desde la sesión o cookie
                if (!$cart_token) {
                    $cart_token = StarterAI::get_cart_token();
                    update_post_meta($order_id, '_cart_token', $cart_token);
                }

                // Obtener datos de facturación
                $billing_data = [
                    'first_name'  => $order->get_billing_first_name(),
                    'last_name'   => $order->get_billing_last_name(),
                    'email'       => $order->get_billing_email(),
                    'phone'       => $order->get_billing_phone(),
                    'address'     => $order->get_billing_address_1(),
                    'city'        => $order->get_billing_city(),
                    'state'       => $order->get_billing_state(),
                    'postcode'    => $order->get_billing_postcode(),
                    'country'     => $order->get_billing_country()
                ];

                // Obtener datos de envío (si están vacíos, usamos los de facturación)
                $shipping_data = [
                    'address'  => $order->get_shipping_address_1() ?: $billing_data['address'],
                    'city'     => $order->get_shipping_city() ?: $billing_data['city'],
                    'state'    => $order->get_shipping_state() ?: $billing_data['state'],
                    'postcode' => $order->get_shipping_postcode() ?: $billing_data['postcode'],
                    'country'  => $order->get_shipping_country() ?: $billing_data['country']
                ];

                // 🔥 Obtener el código de despacho y URL de seguimiento desde los metadatos del pedido
                $coddespacho = get_post_meta($order_id, '_shipex_coddespacho', true);
                $tracking_url = get_post_meta($order_id, '_shipex_url_tracking', true);

                // Procesar los productos del pedido
                $cart_items = [];
                foreach ($order->get_items() as $item) {
                    $product = $item->get_product();
                    if ($product) {
                        $cart_items[] = [
                            "product_id"        => $product->get_id(),
                            "name"              => $product->get_name(),
                            "sku"               => $product->get_sku(),
                            "quantity"          => $item->get_quantity(),
                            "weight"            => $product->get_weight(),
                            "price"             => number_format($product->get_price(), 2, '.', ''),
                            "requires_shipping" => $product->needs_shipping()
                        ];
                    }
                }

                // 🔥 Construir el JSON con los datos del pedido
                $order_data = [
                    "order_id"      => strval($order_id),
                    "cart_token"    => $cart_token,
                    "billing"       => $billing_data,
                    "shipping"      => $shipping_data,
                    "products"      => $cart_items,
                    "order_total"   => number_format($order->get_total(), 2, '.', ''),
                    "payment_method" => $order->get_payment_method_title(),
                    "shipping_method" => $order->get_shipping_method() ?: "No requerido",
                    "date_created"  => $order->get_date_created()->date('Y-m-d H:i:s'),
                    "order_status"  => $to_status,
                    "coddespacho"   => $coddespacho ?: "No disponible",  // 🔥 Agregar código de despacho
                    "tracking_url"  => $tracking_url ?: "No disponible" // 🔥 Agregar URL de seguimiento
                ];

                // 🔹 Obtener la URL del endpoint de `order_update`
                $api_url = StarterAI_Config::get_api_endpoint('order_create');

                if ($api_url) {
                    // 🔥 Enviar datos a la API usando la misma lógica de `add_cart`
                    StarterAI_Logger::send_data_to_endpoint('order_create', $order_data);
                } else {
                    error_log("❌ No se encontró una URL válida para order_create.");
                }

                // 🔹 Registrar logs si está habilitado
                if (get_option('avw_habilitar_logs', 'no') === 'yes') {
                    wc_get_logger()->info(
                        'PEDIDO ACTUALIZADO :: /OrderRegistro :: ' . json_encode($order_data),
                        ['source' => 'asistentevirtualwoocommerce']
                    );
                }
            },
            10,
            4
        );
    }


    public static function get_abandoned_cart_url($cart_token)
    {
        // Verificar que el cart_token no esté vacío
        if (empty($cart_token)) {
            error_log("❌ ERROR: Intento de generar una URL de carrito abandonado sin un cart_token válido.");
            return '';
        }

        // Obtener la URL de checkout de la tienda
        $checkout_url = wc_get_checkout_url();

        // Generar la URL de recuperación con redirección al checkout
        $recovery_url = add_query_arg('recover_cart', urlencode($cart_token), $checkout_url);

        error_log("🔗 URL de checkout con carrito abandonado generada: " . $recovery_url);
        return $recovery_url;
    }

    public static function save_cart_data()
    {
        add_action('woocommerce_add_to_cart', function () {
            self::store_cart(); // Guardar carrito cuando se agrega un producto
        });

        add_action('woocommerce_cart_updated', function () {
            self::store_cart(); // Guardar carrito cuando se actualiza
        });
    }

    public static function recover_cart()
    {
        add_action('wp', function () {
            if (isset($_GET['recover_cart'])) {
                $cart_token = sanitize_text_field($_GET['recover_cart']);
                $saved_cart = get_option('avw_cart_' . $cart_token, '');
                $saved_checkout = get_option('avw_checkout_' . $cart_token, '');

                if (!empty($saved_cart)) {
                    $saved_cart = json_decode($saved_cart, true);
                    WC()->cart->empty_cart();

                    foreach ($saved_cart as $cart_item) {
                        WC()->cart->add_to_cart(
                            $cart_item['product_id'],
                            $cart_item['quantity'],
                            $cart_item['variation_id']
                        );
                    }

                    error_log("✅ Carrito restaurado con token: " . $cart_token);
                } else {
                    error_log("❌ No se encontró carrito para el token: " . $cart_token);
                }

                // Si hay datos de checkout guardados, agregarlos a la sesión
                if (!empty($saved_checkout)) {
                    $saved_checkout = json_decode($saved_checkout, true);
                    WC()->session->set('avw_checkout_data', $saved_checkout);
                    error_log("✅ Datos del checkout restaurados con token: " . $cart_token);
                }
            }
        });
    }


    public static function store_cart()
    {
        $cart_token = StarterAI::get_cart_token();

        // Obtener los productos en el carrito
        $cart_contents = WC()->cart->get_cart();
        $cart_data = [];

        foreach ($cart_contents as $cart_item_key => $cart_item) {
            $cart_data[] = [
                'product_id'   => $cart_item['product_id'],
                'quantity'     => $cart_item['quantity'],
                'variation_id' => isset($cart_item['variation_id']) ? $cart_item['variation_id'] : 0,
            ];
        }

        // Guardar el carrito en la base de datos usando update_option
        update_option('avw_cart_' . $cart_token, json_encode($cart_data));

        error_log("💾 Carrito guardado en BD con token: " . $cart_token);
    }

    public static function get_cart_token()
    {
        if (!WC()->session) {
            return null;
        }

        // Obtener el cart_token desde la sesión
        $cart_token = WC()->session->get('cart_token');

        // Si no está en la sesión, intentamos restaurarlo desde la cookie
        if (empty($cart_token) && isset($_COOKIE['cart_token'])) {
            $cart_token = $_COOKIE['cart_token'];
            WC()->session->set('cart_token', $cart_token);
            error_log("♻️ Cart Token restaurado desde la cookie: " . $cart_token);
        }

        // Si aún no hay un cart_token, generamos uno nuevo
        if (empty($cart_token)) {
            $cart_token = uniqid('cart_', true);
            WC()->session->set('cart_token', $cart_token);
            setcookie('cart_token', $cart_token, time() + (86400 * 30), "/");
            error_log("🆕 Nuevo cart_token generado: " . $cart_token);
        }

        // Asegurar que la cookie y la sesión tengan el mismo valor
        if (!isset($_COOKIE['cart_token']) || $_COOKIE['cart_token'] !== $cart_token) {
            setcookie('cart_token', $cart_token, time() + (86400 * 30), "/");
            error_log("✅ Cart Token sincronizado con la cookie: " . $cart_token);
        }

        return $cart_token;
    }

    public static function restore_cart_token()
    {
        add_action('wp', function () {
            if (isset($_GET['recover_cart']) && !empty($_GET['recover_cart'])) {
                $cart_token = sanitize_text_field($_GET['recover_cart']);

                // 🔹 Evitar regenerar el token si ya existe
                if (!empty($cart_token)) {
                    WC()->session->set('cart_token', $cart_token);
                    setcookie('cart_token', $cart_token, time() + (86400 * 30), "/");
                    error_log("♻️ Carrito abandonado recuperado con token: " . $cart_token);
                }
            }
        });
    }
    
}

// Inicializar la clase StarterAI
StarterAI::init();
